/* Message View -- a component used to render the conversation for Evernote
 * Messages. Written by Andrei Thorp, 2014 <athorp@evernote.com> */

/**
 * Container / namespace for the internal MVC structure of the program.
 *
 * @type {{
 *     Server: *,
 *     Runtime: *,
 *     Template: *,
 *     View: *,
 *     Controller: *,
 *     Helper: *,
 *     Defines: *
 * }}
 */
var Internal = {
    Server: null,
    Runtime: null,
    Template: null,
    View: null,
    Controller: null,
    Helper: null,
    Defines: null
};

/**
 * Container / namespace for the exports for the host environment.
 *
 * @type {*}
 */
var MessageView = {};

/**
 * @typedef {{
 *     id: string,
 *     type: string,
 *     title: string
 * }}
 */
Internal.Attachment;

/**
 * @typedef {{
 *     id: string,
 *     senderID: string,
 *     senderName: string,
 *     senderImage: string,
 *     body: string,
 *     sentAt: string,
 *     attachments: Array.<Internal.Attachment>
 * }}
 */
Internal.Message;

Internal.Defines = function() {
    var me = {};

    /**
     * How much a message must be on screen for it to be considered 'visible'
     *
     * @type {number} percent
     */
    me.MESSAGE_ON_SCREEN_PERCENT = 55;

    /**
     * How long we should wait to decide that the user has stopped scrolling
     *
     * @type {number} milliseconds
     */
    me.SCROLL_JITTER_TIMEOUT = 300;

    /** @type {number} */ me.NODE_ELEMENT_NODE = 1;
    /** @type {number} */ me.NODE_ATTRIBUTE_NODE = 2;
    /** @type {number} */ me.NODE_TEXT_NODE = 3;
    /** @type {number} */ me.NODE_CDATA_SECTION_NODE = 4;
    /** @type {number} */ me.NODE_ENTITY_REFERENCE_NODE = 5;
    /** @type {number} */ me.NODE_ENTITY_NODE = 6;
    /** @type {number} */ me.NODE_PROCESSING_INSTRUCTION_NODE = 7;
    /** @type {number} */ me.NODE_COMMENT_NODE = 8;
    /** @type {number} */ me.NODE_DOCUMENT_NODE = 9;
    /** @type {number} */ me.NODE_DOCUMENT_TYPE_NODE = 10;
    /** @type {number} */ me.NODE_DOCUMENT_FRAGMENT_NODE = 11;
    /** @type {number} */ me.NODE_NOTATION_NODE = 12;

    return me;
}();

Internal.Server = function() {
    var me = {};

    /**
     * Opens the given attachment by id in the host environment.
     *
     * @param {string} id Attachment ID
     *
     * @suppress {undefinedVars|missingProperties}
     */
    me.openAttachment = function(id) {
        console.log('To Server: open attachment', id);

        try {
            Evernote.openAttachment(id);
        } catch (err) {
            console.warn('Couldn\'t run native function:', err.message);
        }
    };

    /**
     * Notifies the host environment that a message has been viewed.
     *
     * @param {string} id Message ID
     *
     * @suppress {undefinedVars|missingProperties}
     */
    me.messageViewed = function(id) {
        console.log('To Server: message viewed', id);

        try {
            Evernote.messageViewed(id);
        } catch (err) {
            console.warn('Couldn\'t run native function:', err.message);
        }
    };

    return me;
}();

Internal.Runtime = function() {
    var me = {};
    /** @type {number} */ me.timeoutID = -1;
    /** @type {Array.<Node>} */ me.lastSeenMessages = [];
    /** @type {Array.<Node>} */ me.currentMessages = [];

    /**
     * Checks if the current platform is Mac.
     *
     * @return {boolean}
     */
    me.isMacPlatform = function() {
        if (window['TEST_WIN']) {
            return false;
        }
        if (window['TEST_MAC']) {
            return true;
        }
        return (navigator.platform.indexOf('Mac') >= 0);
    };

    /**
     * Loads the CSS file for the specified dist target.
     *
     *
     * @param {string} opt_type either 'mac' or 'win'
     */
    me.loadCSS = function(opt_type) {
        /** @type {string} */
        var cssFile = '<link rel="stylesheet" href="css/win-main.css">';

        if (opt_type == null) {
            if (me.isMacPlatform()) {
                opt_type = 'mac';
            } else {
                opt_type = 'win';
            }
        }

        if (opt_type === 'mac') {
            cssFile = '<link rel="stylesheet" href="css/mac-main.css">';
        } else if (opt_type === 'win') {
            cssFile = '<link rel="stylesheet" href="css/win-main.css">';
        }

        /** @type {Node} */ var div = document.createElement('div');
        div.innerHTML = cssFile;
        document.body.appendChild(div.childNodes[0]);
    };

    /**
     * Attempts to load the given image's src based on the value of its
     * data-desired-src attribute. This happens by first trying to background
     * load the image in general, and if that works, then we set the src.
     * If it fails, we load the placeholder image instead.
     *
     * @param {Node} image
     */
    me.tryLoadImage = function(image) {
        /** @type {string} */
        var desiredSrc = image.getAttribute('data-desired-src');

        if (desiredSrc === 'null' ||
            (image.complete === true &&
             image.getAttribute('src') === desiredSrc)) {

            return;
        }

        /** @type {Image} */ var backgroundImage = new Image();

        // Make image visible if it loaded successfully
        image.onload = function _realImageOnLoad() {
            image.style['opacity'] = '1';
        };

        // If backloaded image loaded successfully, then swap in real image
        backgroundImage.onload = function _imageOnLoad() {
            image.src = desiredSrc;
            backgroundImage.onload = null;
            backgroundImage.onerror = null;
        };

        // Try and load the desired image url to a backloaded image
        backgroundImage.src = desiredSrc;
    };

    /**
     * Periodically checks all of the images in the message list to see if
     * their URLs loaded successfully. If they haven't, it forces them to
     * retry. (See README.md for why we do this.)
     */
    me.imageLoadLoop = function() {
        /** @type {Node} */ var messageList;
        /** @type {NodeList} */ var images;
        /** @type {number} */ var timerID;

        timerID = setInterval(function _imageLoadLoop() {
            messageList = Internal.Helper.getMessageList();

            // Probably means we're running unit tests, but best to just bail
            // out if things get this crazy in general.
            if (messageList == null) {
                console.warn('Couldn\'t find messageList, exiting imageLoadLoop');
                clearInterval(timerID);
                return;
            }

            images = messageList.querySelectorAll('img');

            for (var i = images.length - 1; i >= 0; i--) {
                me.tryLoadImage(images[i]);
            }
        }, 5000);
    };

    /**
     * Looks for changes in the window's scroll position and intelligently
     * informs the native client when a new message is viewed.
     */
    me.handleOnScreenMessages = function() {
        /** @type {Node} */ var msg = null;
        clearTimeout(me.timeoutID);

        me.timeoutID = setTimeout(function() {
            me.currentMessages = Internal.View.getOnScreenMessages();
            if (me.currentMessages == null) {
                return;
            }

            // Count from bottom to top so that latest message is reported
            // as viewed before previous ones. Helps simplify native logic.
            for (var i = me.currentMessages.length - 1; i >= 0; i--) {
                msg = me.currentMessages[i];

                if (me.lastSeenMessages.indexOf(msg) === -1) {
                    var messageID = msg.getAttribute('data-message-id');
                    Internal.Server.messageViewed(messageID);
                }
            }

            me.lastSeenMessages = me.currentMessages;
        }, Internal.Defines.SCROLL_JITTER_TIMEOUT);
    };

    return me;
}();

Internal.Helper = function() {
    var me = {};

    /**
     * The index of node relative to parent node.
     *
     * @param {Node} node
     * @return {number}
     */
    me.index = function(node) {
        var ret = 0;
        while ((node = node.previousSibling)) {
            ret++;
        }
        return ret;
    };

    /**
     * Returns the message list element.
     *
     * @return {Node}
     */
    me.getMessageList = function() {
        var list = document.querySelector('#message-list');

        if (list == null) {
            console.warn('Couldn\'t find message list container!');
            return null;
        }

        return list;
    };

    /**
     * Adds a bunch of test data to the message list.
     * FIXME:dholtwick:2014-10-28 These demos and data should be moved
     * out of the main.js file
     *
     * @param {number} opt_startID number to start the fake message id count
     * @param {boolean} opt_batched if set true, uses setMessages()
     */
    me.runDemo = function(opt_startID, opt_batched) {
        if (opt_startID == null) {
            opt_startID = 0;
        }

        if (opt_batched == null) {
            opt_batched = false;
        }

        // Messages are out of order to test the chronological ordering code
        var messages = [
        { id: '' + (opt_startID + 1),
            senderID: 'senderid' + (opt_startID + 1),
            senderName: 'Sir Longnameington the 42352082th in the line of ' +
            'The Lords of Longnamesbury, esquire ' + (opt_startID + 1),
            senderImage: 'img/test/person2.png',
            body: 'Accelerator photo sharing business school drop ' +
            'out <a ramen hustle ' +
            'crush it revenue traction platforms. Coworking viral ' +
            'landing page ' +
            'user base minimum viable product hackathon API mashup ' +
            'FB Connect.',
            sentAt: '2014-10-25, 12:52 PM PST',
            timestamp: 1000 * (opt_startID + 1),
            attachments: [
        ]},
        { id: '' + (opt_startID + 3),
            senderID: 'senderid' + (opt_startID + 3),
            senderName: 'Ianto Jones ' + (opt_startID + 3),
            senderImage: 'img/test/ianto.png',
            body: 'There\'s stuff we don\'t know about. ' +
            'That\'s how Jack likes it. :)',
            sentAt: '7 PM',
            timestamp: 1000 * (opt_startID + 3),
            attachments: [
                { id: 'x', type: 'note', title: 'Secrets' },
                { id: 'y', type: 'notebook', title:
                    'Book of Secrets, interesting in content as well as ' +
                    'length. It goes blah blah blah blah blah blah blah blah ' +
                    'blah blah blah blah blah blah blah blah. And more ' +
                    'importantly: blah blah blah blah blah. What, you thought ' +
                    'I would give out the secrets?'
                }
        ]},
        { id: '' + (opt_startID + 2),
            senderID: 'senderid' + (opt_startID + 2),
            senderName: 'Michelle Remington ' + (opt_startID + 2),
            body: '<a yyy> Main differentiators business model micro economics ' +
            'marketplace equity augmented reality human computer interaction. ' +
            'Board members super angel preferred stock.',
            sentAt: '6/8/14',
            timestamp: 1000 * (opt_startID + 2),
            attachments: [
                { id: '1', type: 'note', title: 'attachment one' },
                { id: '2', type: 'notebook', title: 'attachment two' },
                { id: '3', type: 'notebook', title: 'attachment two' },
                { id: '4', type: 'notebook', title: 'attachment three' },
                { id: '5', type: 'note', title: 'attachment four' },
                { id: '6', type: 'notebook', title: 'attachment five' }
        ]},
        { id: '' + opt_startID,
            senderID: 'senderid' + opt_startID,
            senderName: 'Jennifer Holton ' + opt_startID,
            senderImage: 'img/test/person1.png',
            body: 'Main differentiators business model micro economics ' +
            'https://docs.google.com/spreadsheets/d/1rwnwr3dmidKFYYzyoYO7ZE5CghBfDvTmRKnXfl8WRA/edit#gid=126311555fdgfgdgfdgfhsfghsdhfgsdhfgsdhfgshfhsdhgfdshg' +
            'marketplace equity <b>augmented reality</b> human computer ' +
            'interaction. <br /><br /><a href="http://google.com" title="Google! &lt;">' +
            'Board members super <angel preferred stock.</a>' +
            '<script>alert("hacked, sucker"); console.warn("<a href="javascript:alert(666);">HAX</a>");</script>' +
            'WWWWWWWWWWWWWWWWWWWWWWWWWWWWWWWWWWWWWWWWWWWWWWWWWWWWWWWWWWWWWWWWWWWWWWWWWWWWWWWWWWWWWWWWWWWWWWWWWWWWWWWWWWWWWWWWWWWWW',
            sentAt: 'Yesterday, 2 PM',
            timestamp: 1000 * opt_startID,
            attachments: [
                { id: '1', type: 'note', title: 'attachment one' },
                { id: '2', type: 'note', title: 'attachment two' }
        ]}
        ];

        if (opt_batched) {
            MessageView.setMessages(messages);
        } else {
            for (var i = 0; i < messages.length; i++) {
                MessageView.setMessage(messages[i]);
            }
        }
    };

    /**
     * A longer demo script that adds a whole bunch of messages to the list.
     */
    me.runLongDemo = function() {
        for (var i = 0; i < 20; i++) {
            me.runDemo(i * 4, true);
        }
    };

    return me;
}();

Internal.Template = function() {
    var me = {};

    /**
     * Returns a document fragment representing the specified template.
     *
     * @param {string} templateName
     * @return {DocumentFragment}
     */
    me.getTemplate = function(templateName) {
        var template = document.querySelector('#' + templateName);
        var frag = document.createDocumentFragment();
        var tmpDiv = document.createElement('div');

        if (template == null) {
            return null;
        }

        // FIXME TAKE BETTER CARE AROUND SCRIPT INJECTION
        tmpDiv.innerHTML = template.innerHTML;

        while (tmpDiv.hasChildNodes()) {
            // We only accept elements in templates. Stuff like text nodes
            // are filtered.
            if (tmpDiv.firstChild.nodeType !== Node.ELEMENT_NODE) {
                tmpDiv.removeChild(tmpDiv.firstChild);
            } else {
                frag.appendChild(tmpDiv.firstChild);
            }
        }

        return frag;
    };

    /**
     * Extracts the attachment template from the DOM and returns a version
     * that's ready for editing.
     *
     * @param {!Internal.Attachment} attachment
     * @return {DocumentFragment}
     */
    me.makeAttachment = function(attachment) {
        /** @type {DocumentFragment} */
        var template = me.getTemplate('attachment-template');
        /** @type {Node} */
        var mainDiv = null;

        // TODO:athorp:2014-08-12 enforce that type must be {note,notebook}
        if (template == null ||
            attachment == null ||
            attachment.id == null ||
            attachment.type == null) {

            console.error('Couldn\'t make attachment template!');
            return null;
        }

        attachment.title = attachment.title || 'Untitled';
        mainDiv = template.querySelector('.attachment');

        mainDiv.setAttribute('data-attachment-id', attachment.id);
        mainDiv.classList.add(attachment.type);
        mainDiv.textContent = attachment.title;

        mainDiv.onclick = function() {
            Internal.Server.openAttachment(attachment.id);
        };

        return template;
    };

    var whiteListSchemes = ['http', 'https', 'mailto', 'ftp'];

    /**
     * A white list would be nicer, but this works as well.
     * Inspired by https://github.com/punkave/sanitize-html
     *
     * @param {string} originalHref
     * @return {string}
     */
    var sanitizeHyperlink = function(originalHref) {
        var href = originalHref;

        // So we don't get faked out by a hex or decimal escaped javascript URL #1
        href = decodeURIComponent(href) || href;

        // Browsers ignore character codes of 32 (space) and below in a surprising
        // number of situations. Start reading here:
        // https://www.owasp.org/index.php/XSS_Filter_Evasion_Cheat_Sheet#Embedded_tab
        href = href.replace(/[\x00-\x20]+/, '');

        // Clobber any comments in URLs, which the browser might
        // interpret inside an XML data island, allowing
        // a javascript: URL to be snuck through
        href = href.replace(/<\!\-\-.*?\-\-\>/g, '');

        // Case insensitive so we don't get faked out by JAVASCRIPT #1
        var matches = href.match(/^([a-zA-Z]+)\:/);
        if (!matches) {
            return originalHref || '#';
        }
        var scheme = matches[1].toLowerCase();
        if (whiteListSchemes.indexOf(scheme) === -1) {
            return '#';
        }
        return originalHref.trim() || '#';
    };

    /**
     * Attempts to safely load the given string into the given element, while
     * avoiding script injection and respecting the whitelist of elements
     * that we support, and their attributes.
     *
     * @param {Node} node
     * @param {string} textContent
     */
    me.transformAndAppendContent = function(node, textContent) {
        /** @type {!Element} */
        var root = document.createElement('div');
        /** @type {boolean} */
        var success = true;

        var parts = [textContent];

        /**
         * Walks over parts.
         *
         * @param {Function} fn
         * @return {Array}
         */
        var transformParts = function(fn) {
            var newParts = [];
            for (var i = 0; i < parts.length; i++) {
                var part = parts[i];
                var result = typeof part === 'string' ? fn(part) : part;
                if (result) {
                    newParts = newParts.concat(result);
                }
            }
            return newParts;
        };

        /**
         *
         * @param {RegExp} rx
         * @param {Function} fn
         */
        var transform = function(rx, fn) {
            parts = transformParts(function(part) {
                // var rx = /\n|<br\/?>(<\/br>)?/gi;
                var list = [];

                var m;
                var left = part;
                var i = 0;

                // Reset the regular expression
                rx.lastIndex = 0;

                while (m = rx.exec(left)) {
                    // console.log(i++, 'm', m, m.index, left, list);
                    if (m.index > 0) {
                        // console.log('left:', left.substring(0, m.index));
                        list.push(left.substring(0, m.index));
                    }
                    var element = fn(m);
                    if (element || element === '') {
                        // console.log('el:', element);
                        list.push(element);
                    }
                    else if (element == null) {
                        list.push(m[0]);
                    }
                    left = left.substring(m.index + m[0].length, left.length);
                    // console.log('over:', left, list);

                    // Reset the regular expression
                    rx.lastIndex = 0;

                }
                if (left) {
                    list.push(left);
                }
                return list;
            });
        };

        // Transform \n and <br>
        transform(/\n|<br\/?>(<\/br>)?/gi, function(m) {
            return document.createElement('br');
        });

        // Transform <a>
        transform(/<a\s+href\s*=\s*\"([^\"]*?)\"\s*>(.*?)<\/a>/gi, function(m) {
            var href = sanitizeHyperlink(m[1]);
            if (href) {
                var element = document.createElement('a');
                element.setAttribute('href', href);
                element.textContent = m[2] || href;
            }
            return element;
        });

        // From https://mathiasbynens.be/demo/url-regex
        // Inspired by https://gist.github.com/dperini/729294
        // Modified!
        var rxWebURL = new RegExp(
            // Word boundaries
            '\\b' +
            // protocol identifier
            '(?:(?:https?|ftp)://)' +
            // user:pass authentication
            '(?:\\S+(?::\\S*)?@)?' +

            // HOST! We want it
            '(' +
            '(?:' + (
                // IP address exclusion
                // private & local networks
                '(?!(?:10|127)(?:\\.\\d{1,3}){3})' +
                '(?!(?:169\\.254|192\\.168)(?:\\.\\d{1,3}){2})' +
                '(?!172\\.(?:1[6-9]|2\\d|3[0-1])(?:\\.\\d{1,3}){2})' +
                // IP address dotted notation octets
                // excludes loopback network 0.0.0.0
                // excludes reserved space >= 224.0.0.0
                // excludes network & broacast addresses
                // (first & last IP address of each class)
                '(?:[1-9]\\d?|1\\d\\d|2[01]\\d|22[0-3])' +
                '(?:\\.(?:1?\\d{1,2}|2[0-4]\\d|25[0-5])){2}' +
                '(?:\\.(?:[1-9]\\d?|1\\d\\d|2[0-4]\\d|25[0-4]))'
                ) +
            '|' + (
                // host name
                '(?:(?:[a-z\\u00a1-\\uffff0-9]-*)*[a-z\\u00a1-\\uffff0-9]+)' +
                // domain name
                '(?:\\.(?:[a-z\\u00a1-\\uffff0-9]-*)*[a-z\\u00a1-\\uffff0-9]+)*' +
                // TLD identifier
                '(?:\\.(?:[a-z\\u00a1-\\uffff]{2,}))') +
            ')' +
            ')' +

            // port number
            '(?::\\d{2,5})?' +
            // resource path
            '(?:/\\S*)?' +
            '\\b'
            ,
            'gi'
        );

        // Transform http(s), ftp
        transform(rxWebURL, function(m) {
            var element = document.createElement('a');
            var href = sanitizeHyperlink(m[0]);
            if (href) {
                element.setAttribute('href', href);
                element.textContent = href || '#';
                return element;
            }
        });

        // Transform email and me@me.com
        // Inspired by http://www.regular-expressions.info/examples.html
        transform(/\b(?:mailto:)?([a-z0-9!#$%&'*+/=?^_`\{\|\}~-]+(?:\.[a-z0-9!#$%&'*+\/=?^_`{|}~-]+)*@(?:localhost|(?:[a-z0-9](?:[a-z0-9-]*[a-z0-9])?\.)+[a-z0-9](?:[a-z0-9-]*[a-z0-9])?))\b/gi, function(m) {
            var element = document.createElement('a');
            var href = sanitizeHyperlink(m[1]);
            if (href) {
                element.setAttribute('href', 'mailto:' + href);
                element.textContent = href;
                return element;
            }
        });

        // Smile :)
        if (Internal.Runtime.isMacPlatform()) {
            transform(/:\)|:-\)|:=\)/gi, function(m) {
                return '😃';
            });
        }

        // console.log('parts', parts);

        // Merging into DOM
        for (var i = 0; i < parts.length; i++) {
            var part = parts[i];

            if (typeof part === 'string') {
                part = document.createTextNode(part);
            }

            root.appendChild(part);
        }

        if (success === false) {
            console.warn('Failed to safe load content:', textContent);
            node.appendChild(document.createTextNode('Malformed message!'));
            return;
        }

        // Copy everything back into the given node
        while (root.firstChild) {
            node.appendChild(root.firstChild);
        }
    };

    /**
     * Extracts the message template from the DOM and returns a version
     * that's ready for editing.
     *
     * @param {!Internal.Message} message
     * @return {DocumentFragment}
     */
    me.makeMessage = function(message) {
        /** @type {DocumentFragment} */
        var template = me.getTemplate('message-template');

        if (template == null ||
            message == null ||
            message.id == null ||
            message.senderID == null) {

            console.error('Couldn\'t make message template!');
            return null;
        }

        message.senderName = message.senderName || '';
        message.senderImage = message.senderImage || 'null';
        message.body = message.body || '';
        message.sentAt = message.sentAt || '';
        message.attachments = message.attachments || [];
        message.timestamp = message.timestamp || 1;

        /** @type {Node} */ var main = template.querySelector('.message');
        /** @type {Node} */ var image = template.querySelector('.senderImage');
        /** @type {Node} */ var name = template.querySelector('.senderName');
        /** @type {Node} */ var body = template.querySelector('.body');
        /** @type {Node} */ var sentAt = template.querySelector('.sentAt');
        /** @type {Node} */ var attachments = template.querySelector('.attachments');

        main.setAttribute('data-message-id', message.id);
        image.setAttribute('data-desired-src', message.senderImage);
        Internal.Runtime.tryLoadImage(image);
        name.textContent = message.senderName;
        name.setAttribute('data-sender-id', message.senderID);
        me.transformAndAppendContent(body, message.body);
        sentAt.textContent = message.sentAt;
        sentAt.setAttribute('data-server-timestamp', message.timestamp);

        for (var i = 0; i < message.attachments.length; i++) {
            attachments.appendChild(me.makeAttachment(message.attachments[i]));
        }

        return template;
    };

    return me;
}();

Internal.View = function() {
    var me = {};

    /**
     * Scrolls to the bottom of the message list.
     */
    me.scrollToBottom = function() {
        // FIXME believe it or not, document.body.scrollHeight is a major
        // bottleneck when I profiled this code by loading a massive amount of
        // messages at once. I get a 10x speedup when I remove it here.
        // I should de-jitter this function when it's called repeatedly.
        window.scrollTo(0, document.body.scrollHeight);
    };

    /**
     * Returns a reference to the message DOM element if it's already
     * been rendered. If it hasn't already been rendered, then this returns null.
     *
     * @param {string} messageID
     * @return {?Node}
     */
    me.getMessageDOMByID = function(messageID) {
        /** @type {Node} */ var messageList = Internal.Helper.getMessageList();
        /** @type {Node} */ var childNode;

        for (var i = messageList.childNodes.length - 1; i >= 0; i--) {
            childNode = messageList.childNodes[i];

            if (childNode == null ||
                    childNode.classList == null ||
                    childNode.classList.contains('message') === false) {
                console.warn('Message list contains weird node,', childNode);
                continue;
            }

            if (childNode.getAttribute('data-message-id') === messageID) {
                return childNode;
            }
        }

        return null;
    };

    /**
     * Returns whether the given message element is currently scrolled into view.
     *
     * @param {Node} message
     * @return {boolean}
     */
    me.isOnScreen = function(message) {
        var scrollTop = window.scrollY;
        var scrollBottom = scrollTop + window.innerHeight;

        // Normalized to be relative to the top of window, rather than viewport
        var elementTop = message.getBoundingClientRect().top + scrollTop;
        var elementBottom = message.getBoundingClientRect().bottom + scrollTop;

        // Generate "common interval" between the two "lines"
        var commonTop = Math.max(scrollTop, elementTop);
        var commonBottom = Math.min(scrollBottom, elementBottom);

        // Percentage that the message is on screen
        var overlapPercent = Math.max(0,
                Math.round((commonBottom - commonTop) /
                    (elementBottom - elementTop) * 100));

        return (overlapPercent >= Internal.Defines.MESSAGE_ON_SCREEN_PERCENT);

    };

    /**
     * Returns a list of messageIDs for messages that are currently on screen.
     *
     * @return {Array.<string>}
     */
    me.getOnScreenMessages = function() {
        var messagelist = Internal.Helper.getMessageList();
        if (messagelist == null) {
            console.warn('Cant check on screen messages. Maybe in test env?');
            return [];
        }
        var messages = messagelist.childNodes;
        var onScreen = [];

        for (var i = 0; i < messages.length; i++) {
            if (me.isOnScreen(messages[i])) {
                onScreen.push(messages[i]);
            }
        }

        return onScreen;
    };

    /**
     * Returns the server timestamp of the given element.
     *
     * @param {!(Node|DocumentFragment)} message
     * @return {number}
     */
    var getTimestamp = function(message) {
        var sentAtField = message.querySelector('.sentAt');
        var timestampStr = sentAtField.getAttribute('data-server-timestamp');
        return parseInt(timestampStr, 10);
    };

    /**
     * Adds a message to the message list, ordered by the timestamp field.
     *
     * @param {!(Node|DocumentFragment)} message
     */
    me.addMessage = function(message) {
        var messagelist = Internal.Helper.getMessageList();
        if (messagelist == null) {
            console.warn('Cant check on screen messages. Maybe in test env?');
            return [];
        }
        var messages = messagelist.childNodes;
        var myTimestamp = getTimestamp(message);

        // Special case according to spec where we just always put it at bottom
        if (myTimestamp === -1) {
            messagelist.appendChild(message);
            return;
        }

        var nearestAboveTime = 99999999999;
        var nearestAboveNode = null;

        // TODO:athorp:2014-10-15 this would be better done as a binary search
        for (var i = 0; i < messages.length; i++) {
            var timestamp = getTimestamp(messages[i]);

            if (timestamp > myTimestamp && timestamp < nearestAboveTime) {
                nearestAboveTime = timestamp;
                nearestAboveNode = messages[i];
            }
        }

        if (nearestAboveNode != null) {
            messagelist.insertBefore(message, nearestAboveNode);
        } else {
            messagelist.appendChild(message);
        }
    };

    /**
     * Scrolls the view to show the specified message.
     *
     * @param {string} messageID
     * @param {boolean} opt_bottom
     */
    me.scrollToMessage = function(messageID, opt_bottom) {
        /** @type {Node} */ var message = me.getMessageDOMByID(messageID);

        if (opt_bottom == null) {
            opt_bottom = false;
        }

        if (message == null) {
            console.error('No message with ID', messageID);
            return;
        }

        message.scrollIntoView(!opt_bottom);
    };

    /**
     * Wipes entire display, removing everything.
     */
    me.clear = function() {
        var list = Internal.Helper.getMessageList();

        // A while loop is considerably faster than innerHTML = ''
        while (list.firstChild) {
            list.removeChild(list.firstChild);
        }
    };

    return me;
}();

Internal.Controller = function() {
    var me = {};

    /**
     * Creates and inserts a batch of Message objects.
     *
     * @param {Object} messages Array of Message objects
     */
    var setMessages = function(messages) {
        var frags = [];

        for (var i = 0; i < messages.length; i++) {
            var message = messages[i];
            var messageDOM = Internal.View.getMessageDOMByID(message.id);
            var frag = Internal.Template.makeMessage(message);

            if (messageDOM == null) {
                frags.push(frag);
                continue;
            }

            // Handle updating existing messages
            if (messageDOM != null && messageDOM.parentNode != null) {
                // FIXME:athorp:2014-10-15 this should be the way that messages
                // get updated. Unfortunately, I've had to make a hack around
                // a Mac bug for now. See https://evernote.jira.com/browse/CE-471
                if (Internal.Runtime.isMacPlatform()) {
                    var oldSentAtField = messageDOM.querySelector('.sentAt');
                    var newSentAtField = frag.querySelector('.sentAt');
                    var newTimestamp = newSentAtField.getAttribute('data-server-timestamp');
                    oldSentAtField.textContent = newSentAtField.textContent;
                    oldSentAtField.setAttribute('data-server-timestamp', newTimestamp);

                    // Running this to get it to reorder the message as needed
                    messageDOM.parentNode.removeChild(messageDOM);
                    Internal.View.addMessage(messageDOM);

                } else {
                    // On windows, we don't need to do any hacks
                    messageDOM.parentNode.removeChild(messageDOM);
                    Internal.View.addMessage(frag);
                }
            } else {
                console.warn('Couldn\'t replace message!', messageDOM, frag);
            }
        }

        for (var i = 0; i < frags.length; i++) {
            Internal.View.addMessage(frags[i]);
        }
        Internal.View.scrollToBottom();

        Internal.Runtime.handleOnScreenMessages();
        return 'ok';
    };

    /**
     * Creates and inserts a message given a Message object.
     * Public interface.
     *
     * @param {Object|string} message
     * @return {string}
     */
    me.setMessage = function(message) {
        if (message == null) {
            console.error('Message required!');
            return 'Error: Message required!';
        }

        if (typeof message === 'string') {
            message = /** @type {Object} */ (JSON.parse(message));
        }

        return setMessages([message]);
    };

    /**
     * Batched version of setMessage. I.e. takes an array of Message objects
     * rather than just a single one.
     *
     * @param {Object|string} messages
     */
    me.setMessages = function(messages) {
        if (messages == null) {
            console.error('Messages required!');
        }

        if (typeof messages === 'string') {
            messages = /** @type {Object} */ (JSON.parse(messages));
        }

        return setMessages(messages);
    };

    return me;
}();

// Exports
MessageView.setMessage = Internal.Controller.setMessage;
MessageView.setMessages = Internal.Controller.setMessages;
MessageView.clear = Internal.View.clear;
MessageView.scrollToMessage = Internal.View.scrollToMessage;

// Startup
Internal.Runtime.loadCSS();
Internal.Runtime.imageLoadLoop();
window.addEventListener('scroll', Internal.Runtime.handleOnScreenMessages);
window.addEventListener('resize', Internal.Runtime.handleOnScreenMessages);
